#!/usr/bin/env python3
import ufw.frontend
import ufw.common
import ufw.parser
import ufw.util
import gettext
import os
import json
import re
import time

# --- 下面是从你提供的代码里简化提取的相关函数 ---


def _init_gettext():
    progName = ufw.common.programName
    gettext.install(progName)


def _update_ufw_dependencies():
    _init_gettext()
    frontend = ufw.frontend.UFWFrontend(dryrun=False)
    return frontend, frontend.backend


def get_rules():
    _init_gettext()
    frontend, backend = _update_ufw_dependencies()

    rules = backend.get_rules()
    count = 1
    app_rules = {}
    return_rules = {}
    for r in rules:
        if r.dapp != "" or r.sapp != "":
            tupl = r.get_app_tuple()
            if tupl in app_rules:
                continue
            else:
                app_rules[tupl] = True

        if r.forward:
            rstr = "route {}".format(ufw.parser.UFWCommandRouteRule.get_command(r))
        else:
            rstr = ufw.parser.UFWCommandRule.get_command(r)

        return_rules[count] = rstr
        count += 1
    return return_rules


def parse_rule_string(rule_id, rule_str):
    import re

    result = {
        "id": rule_id,
        "port_start": 0,
        "port_end": 0,
        "protocol": None,
        "direction": "in",
        "action": None,
        "source": None,
        "comment": "",
        "raw": rule_str,
    }

    # 动作 allow / deny / limit ...
    m_action = re.match(r"^(allow|deny|reject|limit)", rule_str)
    if m_action:
        result["action"] = m_action.group(1)

    # 方向
    if " out " in rule_str:
        result["direction"] = "out"

    # 源 IP
    m_ip = re.search(r"\bfrom\s+([^\s]+)", rule_str)
    if m_ip:
        result["source"] = m_ip.group(1)

    # 协议
    m_proto = re.search(r"\bproto\s+(\w+)", rule_str)
    if m_proto:
        result["protocol"] = m_proto.group(1).lower()

    # 注释
    m_comment = re.search(r"comment\s+'([^']+)'", rule_str)
    if m_comment:
        result["comment"] = m_comment.group(1)

    # 端口格式1：9999/tcp 或 1000:2000/udp（可能出现在字符串中任意位置）
    m_port_proto = re.search(r"\b(\d+)(?::(\d+))?/(tcp|udp)\b", rule_str)
    if m_port_proto:
        result["port_start"] = int(m_port_proto.group(1))
        result["port_end"] = (
            int(m_port_proto.group(2))
            if m_port_proto.group(2)
            else result["port_start"]
        )
        result["protocol"] = m_port_proto.group(3).lower()

    # 端口格式2：port 9999 或 port 1000:2000
    m_port = re.search(r"\bport\s+(\d+)(?::(\d+))?", rule_str)
    if m_port:
        result["port_start"] = int(m_port.group(1))
        result["port_end"] = (
            int(m_port.group(2)) if m_port.group(2) else result["port_start"]
        )

    return result


def main():
    rules = get_rules()
    parsed = []

    for rule_id, rule_str in rules.items():
        item = parse_rule_string(rule_id, rule_str)
        parsed.append(item)

    print(json.dumps(parsed, indent=2))


if __name__ == "__main__":
    main()
