#!/bin/bash

config_file="/etc/collectd/collectd.conf"
output_file="/opt/Sinsegye/processes.conf"

# 定义排除的模式列表
declare -a exclude_patterns=(
    "rcu" "irq" "kworker" "ksoftirqd" "migration"
    "udisksd" "sftp-server" "sshd" "systemd" "dbus-daemon"
    "polkitd" "cron" "kthreadd" "watchdog" "kswapd"
    "rsyslogd" "journald" "NetworkManager" "avahi-daemon" "cupsd"
    "ntpd" "dhclient" "rpcbind" "rpc.statd" "lightdm"
    "apache2" "nginx" "postfix" "exim4" "sendmail"
    "dovecot" "vsftpd" "proftpd" "netns" "mm_percpu_wq"
    "cpuhp" "kdevtmpfs" "inet_frag_wq" "kauditd" "khungtaskd"
    "oom_reaper" "writeback" "ksmd" "kintegrityd" "kblockd"
    "blkcg_punt_bio" "tpm_dev_wq" "ata_sff" "md" "edac-poller"
    "devfreq_wq" "rtdm_fd" "ecryptfs-kthrea" "kthrotld" "vfio-irqfd"
    "mld" "ipv6_addrconf" "kstrp" "zswap-shrink" "charger_manager"
    "scsi_" "cryptd" "kdmflush" "raid5wq" "jbd2"
    "ext4-" "ipmi-msghandler" "kaluad" "kmpath_rdacd" "kmpathd"
    "kmpath_handlerd" "multipathd" "wpa_supplicant" "dnsmasq"
    "(sd-pam)" "bash" "sudo" "agetty" "auditd" "cockpit" "gnome"
    "gsd-" "kshark" "kernelshark" "htop" "top" "iotop" "dstat"
    "evolution" "docker" "podman" "card0" "at-spi" "fusermount3"
    "getudev" "goa-" "gpg-" "gvfs-" "gvfsd" "ibus-" "indicator-"
    "journalctl" "libvirtd" "networkd-" "packagekitd" "pipewire"
    "pipewire-" "power-profiles" "pulseaudio" "python3" "rtkit-"
    "ssh-" "switcheroo-cont" "tracker-miner-" "upowerd" "x11vnc"
    "xdg-" "Xorg" "accounts-" "applet" "blueman" "bluetoothd"
    "bolt" "colord" "containerd" "coredump" "crond" "dbus-"
    "deja-dup" "dconf" "dconf-service" "dconfd" "dconf-service"
    "gjs"
)

# 定义需要过滤的最小运行时间（秒）
min_runtime=60

# 初始化结果数组
filtered_processes=()

# 处理进程并填充结果数组
process_filtering() {
    while IFS= read -r line; do
        local process_name=$(echo "$line" | awk '{print $1}')
        local process_time=$(echo "$line" | awk '{print $2}')

        if ((process_time >= min_runtime)); then
            local exclude=false
            for pattern in "${exclude_patterns[@]}"; do
                if [[ "$process_name" == "$pattern"* ]]; then
                    exclude=true
                    break
                fi
            done

            if [ "$exclude" = false ]; then
                filtered_processes+=("$process_name")
            fi
        fi
    done < <(ps -eo comm,etimes --no-headers)
}

# 将结果数组保存到文件中
save_to_file() {
    local file="$1"
    local temp_file="$(mktemp)"

    # 生成文件内容并写入临时文件
    {
        echo "<Plugin processes>"
        for process in "${filtered_processes[@]}"; do
            echo "    Process \"$process\""
        done
        echo "    CollectContextSwitch true"
        echo "    CollectDelayAccounting true"
        echo "    CollectFileDescriptor true"
        echo "    CollectMemoryMaps true"
        echo "</Plugin>"
    } >"$temp_file"

    if [ -f "$file" ]; then
        if ! cmp -s "$temp_file" "$file"; then
            mv "$temp_file" "$file"
            echo "File updated: $file"
            replace_in_config
        else
            echo "No changes detected. File not updated."
            rm "$temp_file"
            return
        fi
    else
        mv "$temp_file" "$file"
        echo "File created: $file"
        replace_in_config
    fi
    systemctl restart collectd
}

# 替换collectd_custom.conf中的目标<Plugin processes>块
replace_in_config() {
    local plugin_block=$(cat "${output_file}")
    local temp_file="$(mktemp)"

    awk -v block="$plugin_block" '
    BEGIN { inside_block = 0 }
    {
        if ($0 ~ /<Plugin processes>/) {
            inside_block = 1
            print block
            next
        }
        if (inside_block && $0 ~ /<\/Plugin>/) {
            inside_block = 0
            next
        }
        if (!inside_block) {
            print $0
        }
    }
    ' "$config_file" >"$temp_file"

    mv "$temp_file" "$config_file"
    echo "Config file updated: $config_file"
}

usage() {
    echo "Usage: $0 [OPTIONS]"
    echo ""
    echo "optional arguments :"
    echo "  --help                 Display this help message"
    echo "  --config <file>        Path to collectd.conf file (default: /etc/collectd/collectd.conf)"
    echo "  --output <file>        Output file for the generated processes block (default: /opt/Sinsegye/processes.conf)"
    exit 0
}

while [[ "$#" -gt 0 ]]; do
    case $1 in
    --help)
        usage
        ;;
    --config)
        config_file="$2"
        shift
        ;;
    --output)
        output_file="$2"
        shift
        ;;
    *)
        echo "Unknown option: $1"
        exit 1
        ;;
    esac
    shift
done

main() {
    process_filtering
    save_to_file "$output_file"
}

main
